﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using System.Linq;

public class OptionsMenu : MonoBehaviour
{
    private Resolution[] _Resolutions;

    public TMP_Dropdown ResolutionsDropdown;
    public TMP_Dropdown QualitiesDropdown;

    public void Start()
    {
        //Setup the UI
        GameObject.Find("FullScreenToggle").GetComponent<Toggle>().isOn = Screen.fullScreen;
        GameObject.Find("MusicSlider").GetComponent<Slider>().value = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.MusicVolume]);
        GameObject.Find("SFXSlider").GetComponent<Slider>().value = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.SFXVolume]);
        SetupResolutions();
        SetupQualities();
    }

    private void Update()
    {
        PageManager.IgnoreBPress = (ResolutionsDropdown.IsExpanded || QualitiesDropdown.IsExpanded);    //If the comboboxes are open, we want to ignore the press so we don't back navigate
    }

    private void SetupResolutions()
    {
        _Resolutions = Screen.resolutions.Where(resolution => resolution.refreshRate == 60).ToArray();  //Only 60Hz resolutions to avoid duplicates
        ResolutionsDropdown.ClearOptions();

        List<string> options = new List<string>();

        int currentResIndex = 0;    //The index of the game's current resolution
        for(int i = 0; i < _Resolutions.Length; i++)
        {
            if (!options.Contains(_Resolutions[i].width + "x" + _Resolutions[i].height))
            {
                //No duplicates, add it
                options.Add(_Resolutions[i].width + "x" + _Resolutions[i].height);
            }

            if (_Resolutions[i].width == Screen.width && _Resolutions[i].height == Screen.height)
            {
                //Found our current resolution
                currentResIndex = options.Count - 1;
            }
        }
        
        ResolutionsDropdown.AddOptions(options);    //Add the resolutions

        //Set the selected value to be the current game resolution
        ResolutionsDropdown.value = currentResIndex;
        ResolutionsDropdown.RefreshShownValue();
    }

    private void SetupQualities()
    {
        QualitiesDropdown.ClearOptions();

        List<string> options = QualitySettings.names.ToList();  //Get all the qualities

        int currentResIndex = 0;
        for (int i = 0; i < QualitySettings.names.Length; i++)
        {
            if(QualitySettings.GetQualityLevel() == i)
            {
                //Found our current quality, break from the loop
                currentResIndex = i;
                break;
            }
        }

        QualitiesDropdown.AddOptions(options);  //Add the qualities

        //Set the selected value to be the current quality
        QualitiesDropdown.value = currentResIndex;
        QualitiesDropdown.RefreshShownValue();
    }

    public void SetResolution(int index)
    {
        //Set the resolution based on the index
        Resolution res = _Resolutions[index];
        Screen.SetResolution(res.width, res.height, Screen.fullScreen);
    }

    public void SetQuality(int index)
    {
        //Set the quality based on the index
        QualitySettings.SetQualityLevel(index, true);
    }

    public void SetFullScreen(bool fullScreen)
    {
        Screen.fullScreen = fullScreen;
    }

    public void SetMusicVolume(float volume)
    {
        //Set the volume, pass true to overwrite
        AudioManager.Instance.SetMusicVolume(volume, true);
    }

    public void SetSFXVolume(float volume)
    {
        //Set the volume, pass true to overwrite
        AudioManager.Instance.SetSFXVolume(volume, true);
    }
}
